#!/bin/bash

declare -a tools=( git tar xz gcc wget make )

download_file() {
	wget $1
	rc=$?
	if [[ $rc != 0 ]] ; then
		echo Failed to download $1
		exit 1
	fi
}

check_tools() {
	echo =======================
	echo Checking required tools
	echo =======================
	for i in "${tools[@]}"
	do
		which $i > /dev/null 2>&1
		rc=$?
		if [[ $rc != 0 ]] ; then
			echo $i not found
			exit 1
		fi
	done
}

check_toolchains() {
	echo ==================
	echo Checking toolchain
	echo ==================
	case $1 in
	openwrt)
		/opt/arm-2014.05/bin/arm-none-linux-gnueabi-gcc -v
		rc=$?
		if [[ $rc != 0 ]] ; then
			echo Please install the ARM toolchain
			echo wget ftp://ftp.visionsystems.de/pub/oss/netcom_netcan_plus/am335x-netcom-toolchain.tar.xz
			echo Extract as superuser:
			echo tar xJf am335x-netcom-toolchain.tar.xz -C /
			exit 1
		fi
		;;
	esac
}

download_dependencies() {
	echo =====================
	echo Download dependencies
	echo =====================
	case $1 in
	netcom_arm)
		if [ ! -f "buildroot.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/netcom_netcan_plus/buildroot.tar.xz
		fi

		if [ -d "buildroot" ]; then
			read -p "Buildroot folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr buildroot > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf buildroot.tar.xz  > /dev/null 2>&1
		;;
	modgate_old)
		if [ ! -f "buildroot.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/modgate_plus/buildroot.tar.xz
		fi

		if [ -d "buildroot" ]; then
			read -p "Buildroot folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr buildroot > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf buildroot.tar.xz  > /dev/null 2>&1
		;;
	modgate)
		if [ ! -f "br-2.2.x.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/modgate_plus/br-2.2.x.tar.xz
		fi

		if [ -d "buildroot" ]; then
			read -p "Buildroot folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr buildroot > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf br-2.2.x.tar.xz  > /dev/null 2>&1
		;;
	netbios)
		if [ ! -f "InstallDisk.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/InstallDisk.tar.xz
		fi

		tar xJf InstallDisk.tar.xz  > /dev/null 2>&1

		if [ ! -f "buildroot.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/netbios/buildroot.tar.xz
		fi

		if [ -d "buildroot" ]; then
			read -p "Buildroot folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr buildroot > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf buildroot.tar.xz  > /dev/null 2>&1
		;;
	openwrt)
		if [ ! -f "openwrt-setup.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/openwrt-setup.tar.xz
		fi

		if [ -d "openwrt-setup" ]; then
			read -p "openwrt-setup folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr openwrt-setup > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf openwrt-setup.tar.xz  > /dev/null 2>&1
		;;
	viavpn)
		if [ ! -f "InstallDisk.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/InstallDisk.tar.xz
		fi

		tar xJf InstallDisk.tar.xz  > /dev/null 2>&1

		if [ ! -f "viavpn-setup.tar.xz" ]; then
			download_file ftp://ftp.vscom.de/pub/oss/viavpn-setup.tar.xz
		fi

		if [ -d "viavpn-setup" ]; then
			read -p "viavpn-setup folder exists. Remove? (y/n)" answer
			if [ "$answer" == "y" ]; then
				rm -fr viavpn-setup > /dev/null 2>&1
			else
				echo Exit
				exit 1
			fi
		fi

		tar xJf viavpn-setup.tar.xz  > /dev/null 2>&1
		;;
	esac

}

setup_buildroot() {
	echo =================
	echo Prepare buildroot
	echo =================
	case $1 in
	netcom_arm)
		cd buildroot
		make BR2_EXTERNAL=../netcom-netcan-bsp vscom_vscs_am335x_linux_5_defconfig
		;;
	modgate)
		cd buildroot
		make BR2_EXTERNAL=../modgate-bsp modgate_defconfig
		;;
	modgate_old)
		cd buildroot
		make BR2_EXTERNAL=../modgate-bsp modgate_old_defconfig
		;;
	netbios)
		cd buildroot
		make BR2_EXTERNAL=../netbios-bsp netbios_defconfig
		;;
	openwrt)
		echo Use the build instructions under OpenWrt Setup from README.md
		;;
	viavpn)
		echo Use the build instructions under viaVPN Setup from README.md
		;;
	esac
}

case $1 in
netcom_arm)
	echo NetCom/NetCAN Plus ARM based BSP
	;;
modgate)
	echo ModGate Plus BSP \(Versions 2.2.x\)
	;;
modgate_old)
	echo ModGate Plus BSP \(Old versions 2.0.x \& 2.1.x\)
	;;
netbios)
	echo Netbios BSP \(1st \& 2nd stage bootloader for viaVPN\)
	;;
viavpn)
	echo viaVPN Setup
	;;
openwrt)
	echo OpenWrt Setup
	;;
clean)
	echo Cleanup
	rm -fr buildroot*
	rm -fr openwrt-setup
	exit 0
	;;
*)
	echo Unknown model
	exit 1
	;;
esac

check_tools

check_toolchains $1

download_dependencies $1

setup_buildroot $1
